/*:
 * @target MZ
 * @plugindesc ピクチャ・カットイン++（サブフォルダ対応／退場方向指定／ボヨン強化／回転ゆらゆら）
 * @author you
 *
 * @command CutIn
 * @text カットイン開始
 * @desc 指定ピクチャがボヨン入場→待機(揺れ可)→ボヨン退場→自動消去
 *
 * @arg pictureId
 * @text ピクチャID
 * @type number
 * @min 1
 * @default 1
 *
 * @arg useExisting
 * @text 既存ピクチャを使う
 * @type boolean
 * @on 使う（表示済み）
 * @off 使わない（ここで表示）
 * @default false
 *
 * @arg imageFile
 * @text 画像ファイル（UI選択）
 * @type file
 * @dir img/pictures
 * @desc 従来のUI選択。サブフォルダは未対応。下の「画像パス」が空のときに使われます。
 * @default
 *
 * @arg imagePath
 * @text 画像パス（サブフォルダ可）
 * @type string
 * @desc 例）face/ok  => img/pictures/face/ok.png（指定があればこちらを優先）
 * @default
 *
 * @arg origin
 * @text 原点
 * @type select
 * @option 左上(0) @value 0
 * @option 中心(1) @value 1
 * @default 1
 *
 * @arg targetX
 * @text 目標X
 * @type number
 * @default 0
 *
 * @arg targetY
 * @text 目標Y
 * @type number
 * @default 0
 *
 * @arg enterDirection
 * @text 入場方向
 * @type select
 * @option Left @value left
 * @option Right @value right
 * @option Top @value top
 * @option Bottom @value bottom
 * @default left
 *
 * @arg exitDirection
 * @text 退場方向
 * @type select
 * @option Auto(入場の逆) @value auto
 * @option Left @value left
 * @option Right @value right
 * @option Top @value top
 * @option Bottom @value bottom
 * @default auto
 *
 * @arg margin
 * @text 画面外開始/退場マージン(px)
 * @type number
 * @min 0
 * @default 80
 *
 * @arg enterDuration
 * @text 入場時間(フレーム)
 * @type number
 * @min 1
 * @default 18
 *
 * @arg stayDuration
 * @text 待機時間(フレーム)
 * @type number
 * @min 0
 * @default 45
 *
 * @arg exitDuration
 * @text 退場時間(フレーム)
 * @type number
 * @min 1
 * @default 18
 *
 * @arg baseScale
 * @text 基本スケール(%)
 * @type number
 * @min 1
 * @default 100
 *
 * @arg boingAmp
 * @text ボヨン強度(入退場オーバー率%)
 * @type number
 * @min 100
 * @default 125
 *
 * @arg boingTimes
 * @text 弾み回数(入場)
 * @type number
 * @min 0
 * @max 6
 * @default 2
 *
 * @arg boingDamping
 * @text 減衰率(入場)
 * @type number
 * @min 0
 * @max 100
 * @default 55
 *
 * @arg exitBoingTimes
 * @text 弾み回数(退場)
 * @type number
 * @min 0
 * @max 6
 * @default 1
 *
 * @arg exitBoingDamping
 * @text 減衰率(退場)
 * @type number
 * @min 0
 * @max 100
 * @default 45
 *
 * @arg opacity
 * @text 不透明度(0-255)
 * @type number
 * @min 0
 * @max 255
 * @default 255
 *
 * @arg blendMode
 * @text 合成
 * @type select
 * @option Normal @value 0
 * @option Add @value 1
 * @option Multiply @value 2
 * @option Screen @value 3
 * @default 0
 *
 * @arg waitForFinish
 * @text 完了までウェイト
 * @type boolean
 * @on する
 * @off しない
 * @default false
 *
 * @arg swayEnable
 * @text 待機中ゆらゆらON
 * @type boolean
 * @on ON
 * @off OFF
 * @default true
 *
 * @arg swayAngle
 * @text 回転角度±(度)
 * @type number
 * @min 0
 * @default 6
 *
 * @arg swayPeriod
 * @text 揺れ周期(フレーム/往復)
 * @type number
 * @min 1
 * @default 30
 *
 * @arg swayScaleAmp
 * @text ついでに拡大縮小±(%)
 * @type number
 * @min 0
 * @default 2
 *
 * @arg swayMoveX
 * @text ついでに左右揺れ(px)
 * @type number
 * @min 0
 * @default 0
 *
 * @arg swayMoveY
 * @text ついでに上下揺れ(px)
 * @type number
 * @min 0
 * @default 0
 */
(() => {
  const pluginName = document.currentScript.src.match(/([^/]+)\.js$/)[1];

  // ---------- helpers ----------
  const N = (v, d=0) => (v === undefined || v === null || v === '') ? d : Number(v);

  // Elasticっぽい"ボヨン"補間（回数・減衰可）
  function elasticOut(t, bounces=2, damping=0.55, overshoot=1.25) {
    // t:0→1, bounces: 弾む回数, damping: 減衰(0-1), overshoot: 初回ピーク倍率
    if (t <= 0) return 0;
    if (t >= 1) return 1;
    const p = t;
    // 基本はOutBackベースで最初にオーバー、その後は減衰サインでバウンド
    const s = 1.70158;
    const back = 1 + s * Math.pow(p - 1, 3) + (p - 1) * (s * s); // outBack
    let bounce = 0;
    if (bounces > 0) {
      const w = Math.PI * (bounces + 0.5); // 0.5で終端が谷になり過ぎない調整
      // 減衰：exp(-k * p) の k を damping から近似
      const k = Math.max(0.001, damping) * 5.0;
      const decay = Math.exp(-k * p);
      bounce = Math.sin(p * w) * decay * (overshoot - 1); // overshootからの相対量
    }
    return Math.min(1, Math.max(0, back + bounce));
  }
  function elasticIn(t, bounces=1, damping=0.45, undershoot=0.85) {
    if (t <= 0) return 0;
    if (t >= 1) return 1;
    const p = t;
    const s = 1.70158;
    const back = p * p * ((s + 1) * p - s); // inBack
    let bounce = 0;
    if (bounces > 0) {
      const w = Math.PI * (bounces + 0.5);
      const k = Math.max(0.001, damping) * 5.0;
      const decay = Math.exp(-k * p);
      bounce = -Math.sin(p * w) * decay * (1 - undershoot); // 縮んで退く方向
    }
    return Math.min(1, Math.max(0, back + bounce));
  }

  function startXYFromDirection(dir, tx, ty, margin) {
    let sx = tx, sy = ty;
    switch (dir) {
      case 'left':  sx = -margin; break;
      case 'right': sx = Graphics.width + margin; break;
      case 'top':   sy = -margin; break;
      case 'bottom':sy = Graphics.height + margin; break;
    }
    return [sx, sy];
  }
  function exitXYFromDirection(dir, tx, ty, margin) {
    let ex = tx, ey = ty;
    switch (dir) {
      case 'left':  ex = -margin; break;
      case 'right': ex = Graphics.width + margin; break;
      case 'top':   ey = -margin; break;
      case 'bottom':ey = Graphics.height + margin; break;
    }
    return [ex, ey];
  }

  // ---------- Game_Picture: state ----------
  Game_Picture.prototype._cutInPlus = null;

  Game_Picture.prototype._cipInit = function(st) {
    this._cutInPlus = st;
    // 初期配置
    this._x = st.sx; this._y = st.sy;
    this._scaleX = st.baseScale * 100;
    this._scaleY = st.baseScale * 100;
    this._opacity = 0;
    this._angle = 0;
  };

  Game_Picture.prototype._cipUpdate = function(pictureId) {
    const st = this._cutInPlus;
    if (!st) return;

    // phase: enter -> stay -> exit
    if (st.phase === 'enter') {
      st.t++;
      const r = Math.min(1, st.t / st.enterDur);
      const e = elasticOut(r, st.boingTimes, st.boingDamp, st.boingOvershoot);
      this._x = st.sx + (st.tx - st.sx) * e;
      this._y = st.sy + (st.ty - st.sy) * e;

      // スケール：着地に向けてbaseへ、ただし弾みを反映
      const scaleE = 1 + (st.boingOvershoot - 1) * (1 - (1 - e) * (1 - e));
      const s = st.baseScale * scaleE;
      this._scaleX = s * 100;
      this._scaleY = s * 100;

      // フェードイン
      this._opacity = Math.round(st.opacity * r);

      if (r >= 1) {
        // 着地補正
        this._x = st.tx; this._y = st.ty;
        this._scaleX = st.baseScale * 100;
        this._scaleY = st.baseScale * 100;
        this._opacity = st.opacity;
        st.phase = st.stayDur > 0 ? 'stay' : 'exit';
        st.t = 0;
      }

    } else if (st.phase === 'stay') {
      st.t++;
      // ゆらゆら（回転＋微スケール＋微移動）
      if (st.swayEnable && st.stayDur > 0) {
        const th = (st.t / st.swayPeriod) * Math.PI; // 0→π→2π…で往復想定
        const sin = Math.sin(th);
        // angle
        this._angle = st.swayAngle * sin;
        // scale
        const s = st.baseScale * (1 + (st.swayScaleAmp / 100) * sin);
        this._scaleX = s * 100;
        this._scaleY = s * 100;
        // move
        this._x = st.tx + st.swayMoveX * sin;
        this._y = st.ty + st.swayMoveY * sin;
      } else {
        // 静止補正
        this._angle = 0;
        this._x = st.tx; this._y = st.ty;
        this._scaleX = st.baseScale * 100;
        this._scaleY = st.baseScale * 100;
      }

      if (st.t >= st.stayDur) {
        st.phase = 'exit';
        st.t = 0;
      }

    } else if (st.phase === 'exit') {
      st.t++;
      const r = Math.min(1, st.t / st.exitDur);

      const e = elasticIn(r, st.exitBoingTimes, st.exitBoingDamp, st.exitUndershoot);
      const [ex, ey] = st.exitPos;
      this._x = st.tx + (ex - st.tx) * e;
      this._y = st.ty + (ey - st.ty) * e;

      // 少し縮みながら退く
      const s = st.baseScale * (1 - (1 - st.exitUndershoot) * e);
      this._scaleX = s * 100;
      this._scaleY = s * 100;

      // フェードアウト
      this._opacity = Math.round(st.opacity * (1 - r));

      // 回転は退場で戻す
      this._angle = (1 - r) * this._angle;

      if (r >= 1) {
        $gameScreen.erasePicture(pictureId);
        this._cutInPlus = null;
      }
    }
  };

  // ---------- Sprite_Picture.update hook ----------
  const _Sprite_Picture_update = Sprite_Picture.prototype.update;
  Sprite_Picture.prototype.update = function() {
    _Sprite_Picture_update.call(this);
    const pic = this.picture();
    if (!pic) return;
    if (pic._cutInPlus) {
      pic._cipUpdate(this._pictureId);
    }
  };

  // ---------- Command ----------
  PluginManager.registerCommand(pluginName, 'CutIn', args => {
    const id = N(args.pictureId, 1);
    const useExisting = args.useExisting === 'true';
    const rawPath = String(args.imagePath || '').trim();
    const uiFile  = String(args.imageFile || '').trim();
    const name    = rawPath || uiFile; // 拡張子は不要。サブフォルダ可（rawPath）
    const origin = N(args.origin, 1);
    const tx = N(args.targetX, 0);
    const ty = N(args.targetY, 0);
    const enterDir = String(args.enterDirection || 'left');
    const exitDirParam = String(args.exitDirection || 'auto');
    const margin = N(args.margin, 80);
    const enterDur = Math.max(1, N(args.enterDuration, 18));
    const stayDur  = Math.max(0, N(args.stayDuration, 45));
    const exitDur  = Math.max(1, N(args.exitDuration, 18));
    const baseScale = Math.max(0.01, N(args.baseScale, 100)) / 100;
    const boingOvershoot = Math.max(1.0, N(args.boingAmp, 125) / 100);
    const boingTimes = N(args.boingTimes, 2);
    const boingDamp  = Math.min(1, Math.max(0, N(args.boingDamping, 55) / 100));
    const exitBoingTimes = N(args.exitBoingTimes, 1);
    const exitBoingDamp  = Math.min(1, Math.max(0, N(args.exitBoingDamping, 45) / 100));
    const exitUndershoot = 0.85; // 退場で少し小さく
    const opacity = Math.max(0, Math.min(255, N(args.opacity, 255)));
    const blendMode = N(args.blendMode, 0);
    const waitForFinish = args.waitForFinish === 'true';

    const swayEnable = args.swayEnable === 'true';
    const swayAngle = N(args.swayAngle, 6);
    const swayPeriod = Math.max(1, N(args.swayPeriod, 30));
    const swayScaleAmp = Math.max(0, N(args.swayScaleAmp, 2));
    const swayMoveX = Math.max(0, N(args.swayMoveX, 0));
    const swayMoveY = Math.max(0, N(args.swayMoveY, 0));

    if (!useExisting) {
      $gameScreen.showPicture(id, name, origin, tx, ty, baseScale*100, baseScale*100, 0, blendMode);
    }
    const pic = $gameScreen.picture(id);
    if (!pic) return;

    // 入場開始XY
    const [sx, sy] = startXYFromDirection(enterDir, tx, ty, margin);

    // 退場方向
    let exitDir = exitDirParam;
    if (exitDir === 'auto') {
      exitDir = (enterDir === 'left') ? 'right' :
                (enterDir === 'right') ? 'left' :
                (enterDir === 'top') ? 'bottom' : 'top';
    }
    const exitPos = exitXYFromDirection(exitDir, tx, ty, margin);

    pic._cipInit({
      phase: 'enter',
      t: 0,
      sx, sy, tx, ty,
      exitPos,
      enterDur, stayDur, exitDur,
      baseScale,
      boingOvershoot,
      boingTimes,
      boingDamp,
      exitBoingTimes,
      exitBoingDamp,
      exitUndershoot,
      opacity,
      swayEnable,
      swayAngle,
      swayPeriod,
      swayScaleAmp,
      swayMoveX,
      swayMoveY
    });

    if (waitForFinish) {
      const total = enterDur + stayDur + exitDur;
      // マップ/バトルで安全にウェイト
      const interp = SceneManager._scene._messageWindow?.$gameMessage ? $gameMap._interpreter : $gameMap._interpreter;
      if (interp) interp.wait(total);
    }
  });
})();
